<?php

namespace App\Filament\Pages;

use Filament\Pages\Page;
use Filament\Actions\Action;
use Filament\Forms\Components\FileUpload;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Schema;

class Backups extends Page
{
    protected static ?string $navigationIcon = 'heroicon-o-arrow-path-rounded-square';
    protected static ?string $navigationLabel = 'النسخ والاستعادة';
    protected static ?string $title = 'مدير النسخ الاحتياطي (سريع)';
    protected static ?string $navigationGroup = 'الإعدادات';

    protected static string $view = 'filament.pages.backups';

    public function getViewData(): array
    {
        return [];
    }

    protected function getHeaderActions(): array
    {
        return [
            // === زر 1: تحميل النسخة (Download) ===
            Action::make('download_backup')
                ->label('تحميل نسخة كاملة (Download)')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('success')
                ->action(function () {
                    try {
                        $tables = DB::select('SHOW TABLES');
                        $backupData = [];

                        foreach ($tables as $table) {
                            $tableName = reset($table);
                            // استبعاد جداول النظام
                            if (in_array($tableName, ['migrations', 'sessions', 'cache', 'cache_locks', 'jobs', 'failed_jobs', 'job_batches', 'password_reset_tokens'])) continue;
                            
                            $backupData[$tableName] = DB::table($tableName)->get()->toArray();
                        }

                        $fileName = 'backup_' . date('Y-m-d_H-i-s') . '.json';
                        $jsonContent = json_encode($backupData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

                        return response()->streamDownload(function () use ($jsonContent) {
                            echo $jsonContent;
                        }, $fileName);

                    } catch (\Exception $e) {
                        Notification::make()->title('فشل التحميل')->body($e->getMessage())->danger()->send();
                    }
                }),

            // === زر 2: استعادة النسخة (Restore) ===
            Action::make('restore_backup')
                ->label('استعادة نسخة (Restore)')
                ->icon('heroicon-o-arrow-up-tray')
                ->color('danger')
                ->requiresConfirmation()
                ->modalHeading('تحذير: سيتم حذف البيانات الحالية!')
                ->modalDescription('سيتم مسح جميع البيانات الحالية واستبدالها بالنسخة المرفوعة.')
                ->form([
                    FileUpload::make('backup_file')
                        ->label('اختر ملف النسخة (.json)')
                        ->acceptedFileTypes(['application/json'])
                        ->disk('local')
                        ->directory('temp')
                        ->required(),
                ])
                ->action(function (array $data) {
                    try {
                        $path = $data['backup_file'];

                        if (!Storage::disk('local')->exists($path)) {
                             throw new \Exception("لم يتم العثور على الملف.");
                        }

                        $jsonContent = Storage::disk('local')->get($path);
                        $backupData = json_decode($jsonContent, true);

                        if (!$backupData || !is_array($backupData)) {
                            throw new \Exception("الملف تالف أو فارغ.");
                        }

                        // --- بدء العملية ---
                        DB::beginTransaction();
                        
                        // إيقاف القيود مؤقتاً لحذف البيانات بأمان
                        DB::statement('SET FOREIGN_KEY_CHECKS=0;');

                        foreach ($backupData as $tableName => $rows) {
                            if (!Schema::hasTable($tableName)) continue;

                            // استخدمنا delete بدلاً من truncate للحفاظ على الـ Transaction
                            DB::table($tableName)->delete();

                            if (!empty($rows)) {
                                $rows = array_map(function ($item) { return (array)$item; }, $rows);
                                
                                // إدخال البيانات
                                foreach (array_chunk($rows, 50) as $chunk) {
                                    DB::table($tableName)->insert($chunk);
                                }
                            }
                        }

                        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
                        DB::commit(); // الآن سيعمل الحفظ بنجاح

                        // تنظيف
                        Storage::disk('local')->delete($path);

                        Notification::make()->title('تمت الاستعادة بنجاح!')->success()->send();

                    } catch (\Exception $e) {
                        // محاولة التراجع فقط إذا كان هناك معاملة نشطة
                        try { DB::rollBack(); } catch (\Exception $t) {}
                        
                        Notification::make()
                            ->title('فشلت الاستعادة')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();
                    }
                }),
        ];
    }
}