<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ProductResource\Pages;
use App\Models\Product;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

class ProductResource extends Resource
{
    protected static ?string $model = Product::class;

    protected static ?string $navigationIcon = 'heroicon-o-cake';
    protected static ?string $navigationGroup = 'إدارة المنيو';
    protected static ?int $navigationSort = 2;

    protected static ?string $modelLabel = 'منتج';
    protected static ?string $pluralModelLabel = 'المنتجات';
    protected static ?string $navigationLabel = 'المنتجات';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('تفاصيل المنتج')
                    ->schema([
                        // 1. القسم
                        Forms\Components\Select::make('category_id')
                            ->relationship('category', 'name')
                            ->label('القسم')
                            ->searchable()
                            ->preload()
                            ->required(),

                        // 2. اسم المنتج
                        Forms\Components\TextInput::make('name')
                            ->label('اسم المنتج')
                            ->required()
                            ->maxLength(255),

                        // 3. وصف المنتج
                        Forms\Components\Textarea::make('description')
                            ->label('وصف المنتج')
                            ->rows(3)
                            ->columnSpanFull()
                            ->helperText('اكتب وصفاً مختصراً يظهر تحت اسم المنتج (مثل المكونات)'),

                        // 4. السعر الأساسي
                        Forms\Components\TextInput::make('price')
                            ->label('السعر الأساسي')
                            ->numeric()
                            ->prefix('SAR')
                            ->live()
                            ->required(fn (Get $get): bool => empty($get('sizes')))
                            ->helperText(fn (Get $get) => empty($get('sizes')) ? 'مطلوب في حال عدم وجود أحجام' : 'اختياري لوجود أحجام'),

                        // 5. الأحجام
                        Forms\Components\Repeater::make('sizes')
                            ->label('أحجام المنتج (اختياري)')
                            ->schema([
                                Forms\Components\TextInput::make('name')
                                    ->label('اسم الحجم (مثلاً: كبير)')
                                    ->required(),
                                Forms\Components\TextInput::make('price')
                                    ->label('السعر')
                                    ->numeric()
                                    ->prefix('SAR')
                                    ->required(),
                            ])
                            ->columns(2)
                            ->grid(1)
                            ->defaultItems(0)
                            ->live()
                            ->columnSpanFull(),

                        // 6. الإضافات (الاستبعادات) - هذا هو الكود الصحيح فقط
                        Forms\Components\CheckboxList::make('exclusionList') // ✅ الاسم الجديد
                            ->relationship('exclusionList', 'name')          // ✅ العلاقة الجديدة
                            ->label('خيارات "بدون" (الاستبعادات)')
                            ->columns(3)
                            ->gridDirection('row')
                            ->bulkToggleable()
                            ->columnSpanFull(),

                        // ⚠️ (تم حذف الكود المكرر والقديم من هنا)

                        // 8. الصورة
                        Forms\Components\FileUpload::make('image')
                            ->label('صورة المنتج')
                            ->image()
                            ->directory('products')
                            ->columnSpanFull(),

                        // 9. الحالة
                        Forms\Components\Toggle::make('is_visible')
                            ->label('متاح للبيع')
                            ->default(true)
                            ->required(),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ImageColumn::make('image')
                    ->label('الصورة')
                    ->circular(),

                Tables\Columns\TextColumn::make('name')
                    ->label('المنتج')
                    ->description(fn (Product $record): string => $record->description ?? '')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('category.name')
                    ->label('القسم')
                    ->sortable()
                    ->badge(),

                Tables\Columns\TextColumn::make('price')
                    ->label('السعر')
                    ->money('SAR')
                    ->sortable()
                    ->formatStateUsing(function ($state, Product $record) {
                        if (empty($state) && !empty($record->sizes)) {
                            $minPrice = collect($record->sizes)->min('price');
                            return 'يبدأ من ' . number_format($minPrice, 2) . ' SAR';
                        }
                        return $state ? number_format($state, 2) . ' SAR' : '-';
                    }),

                Tables\Columns\ToggleColumn::make('is_visible')
                    ->label('الحالة'),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('category')
                    ->relationship('category', 'name')
                    ->label('تصفية حسب القسم'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListProducts::route('/'),
            'create' => Pages\CreateProduct::route('/create'),
            'edit' => Pages\EditProduct::route('/{record}/edit'),
        ];
    }
}